`include "common.v"
`include "psw_checker.v"

/* Mnemonici instrukcija */
`define LDR0 8'b00000000
`define LDR1 8'b00000001
`define LDR2 8'b00000010
`define LDR3 8'b00000011
`define LDR4 8'b00000100
`define LDR5 8'b00000101
`define LDR6 8'b00000110
`define LDR7 8'b00000111
`define LDR8 8'b00001000
`define LDR9 8'b00001001
`define LDR10 8'b00001010
`define LDR11 8'b00001011
`define LDR12 8'b00001100
`define LDR13 8'b00001101
`define LDR14 8'b00001110
`define LDR15 8'b00001111

`define LDC0 8'b00010000
`define LDC1 8'b00010001
`define LDC2 8'b00010010
`define LDC3 8'b00010011
`define LDC4 8'b00010100
`define LDC5 8'b00010101
`define LDC6 8'b00010110
`define LDC7 8'b00010111
`define LDC8 8'b00011000
`define LDC9 8'b00011001
`define LDC10 8'b00011010
`define LDC11 8'b00011011
`define LDC12 8'b00011100
`define LDC13 8'b00011101
`define LDC14 8'b00011110
`define LDC15 8'b00011111

`define STR0 8'b00100000
`define STR1 8'b00100001
`define STR2 8'b00100010
`define STR3 8'b00100011
`define STR4 8'b00100100
`define STR5 8'b00100101
`define STR6 8'b00100110
`define STR7 8'b00100111
`define STR8 8'b00101000
`define STR9 8'b00101001
`define STR10 8'b00101010
`define STR11 8'b00101011
`define STR12 8'b00101100
`define STR13 8'b00101101
`define STR14 8'b00101110
`define STR15 8'b00101111

`define LDI 8'b10110000
`define STI 8'b10110001

`define CMP 8'b10000000
`define TST 8'b10000001
`define ADD 8'b10000010
`define ADC 8'b10000011
`define SUB 8'b10000100
`define INC 8'b10000101
`define DEC 8'b10000110
`define NEG 8'b10000111
`define AND 8'b10001000
`define OR  8'b10001001
`define XOR 8'b10001010
`define NOT 8'b10001011
`define SHL 8'b10001100
`define SHR 8'b10001101
`define SAR 8'b10001110
`define MOV 8'b10001111


`define JMP 8'b01000000
`define JE  8'b01000001
`define JNE 8'b01000010
`define JA  8'b01000011
`define JB  8'b01000100
`define JAE 8'b01000101
`define JBE 8'b01000110
`define JG  8'b01000111
`define JL  8'b01001000
`define JGE 8'b01001001
`define JLE 8'b01001010

`define JMPI 8'b11000000
`define JEI  8'b11000001
`define JNEI 8'b11000010
`define JAI  8'b11000011
`define JBI  8'b11000100
`define JAEI 8'b11000101
`define JBEI 8'b11000110
`define JGI  8'b11000111
`define JLI  8'b11001000
`define JGEI 8'b11001001
`define JLEI 8'b11001010

`define HALT 8'b11111111


/* Stanja konacnog automata kontrolne jedinice */
`define FETCH 4'd0
`define WAIT_BUS_INST 4'd1
`define REQUEST_INST 4'd2
`define WAIT_INST 4'd3
`define DECODE 4'd4
`define EXECUTE 4'd5
`define WRITE 4'd6
`define WAIT_BUS_LOAD 4'd7
`define REQUEST_LOAD 4'd8
`define WAIT_LOAD 4'd9
`define WAIT_BUS_STORE 4'd10
`define REQUEST_STORE 4'd11
`define WAIT_STORE 4'd12
`define HALT_STATE 4'd13


/* Kontrolna jedinica procesora. Radi u ritmu casovnika clk. Na ulazu
   ima komponente IR registra, kao i izlaz PSW registra (koji se
   koristi kod instrukcija skoka). Signali koji pocinju slovom 'e' su
   razni signali kojima se kontrolisu ostale komponente
   procesora. bus_request se salje arbitru magistrale. bus_grant se dobija
   od arbitra magistrale. bus_in_use je signal koji obavestava magistralu
   (i uredjaje na njoj) da je u toku transakcija, dok bus_command signal
   odredjuje tip operacije na magistrali (0: read, 1: write). bus_wait
   signal stize sa magistrale od slave uredjaja u slucaju da je potrebno
   sacekati jos jedan ciklus za zavrsetak operacije. */

module cu(clk, psw_out, ir_op_code, ir_dst_reg, ir_src_reg,
	  e_reg_a_in, e_reg_b_in, e_reg_a_out, e_reg_b_out, 
	  e_alu, 
	  e_ac_alu_in, e_ac_a_out, e_ac_b_out,
	  e_psw_in,
	  e_pc_a_in, e_pc_b_in, e_pc_a_out, e_pc_b_out, e_pc_inc,
	  e_ir_a_in, e_ir_b_in,
	  e_mar_a_in, e_mar_b_in, e_addr_out, e_calc_addr,
	  e_mdr_a_in, e_mdr_b_in, e_data_in, e_mdr_a_out, e_mdr_b_out, e_data_out, e_calc_const,  
	  alu_op, reg_a_sel, reg_b_sel, bus_request, bus_grant, bus_in_use, bus_command, bus_wait, halt_computer);
   
   input clk;
   input [3:0] psw_out;
   input [7:0] ir_op_code;
   input [3:0] 	ir_dst_reg;
   input [3:0] 	ir_src_reg;
   output reg  e_reg_a_in, e_reg_b_in, e_reg_a_out, e_reg_b_out, 
	       e_alu, 
	       e_ac_alu_in, e_ac_a_out, e_ac_b_out,
	       e_psw_in, 
	       e_pc_a_in, e_pc_b_in, e_pc_a_out, e_pc_b_out, e_pc_inc,
	       e_ir_a_in, e_ir_b_in,
	       e_mar_a_in, e_mar_b_in, e_addr_out, e_calc_addr,
	       e_mdr_a_in, e_mdr_b_in, e_data_in, e_mdr_a_out, e_mdr_b_out, e_data_out, e_calc_const;   	       

   output reg[3:0] alu_op;
   output reg[3:0] reg_a_sel;
   output reg[3:0] reg_b_sel;
   output    reg   bus_request, bus_in_use, bus_command;
   input       bus_grant, bus_wait;
   output reg  halt_computer;
   
   
   reg [3:0]   state;
   wire        cond;

   // Kolo psw_checker na osnovu operacionog koda instrukcije skoka
   // i stanja flegova odredjuje da li treba skociti ili ne (izlaz
   // cond je 1 ako treba, a 0 ako ne treba).
   psw_checker _psw_checker(ir_op_code[3:0], psw_out, cond);
   
   /* Inicijalno su svi signali iskljuceni, a pocetno stanje je FETCH */
   initial
     begin
	state <= `FETCH;
	e_reg_a_in <= `DISABLE;
	e_reg_b_in <= `DISABLE;
	e_reg_a_out <= `DISABLE;
	e_reg_b_out <= `DISABLE;
	e_alu <= `DISABLE;
	e_psw_in <= `DISABLE;      
	e_ac_alu_in <= `DISABLE;
	e_ac_a_out <= `DISABLE;
	e_ac_b_out <= `DISABLE;
	e_pc_a_in <= `DISABLE;
	e_pc_b_in <= `DISABLE;
	e_pc_a_out <= `DISABLE;
	e_pc_b_out <= `DISABLE;
	e_pc_inc <= `DISABLE;	
	e_ir_a_in <= `DISABLE;
	e_ir_b_in <= `DISABLE;	
	e_mar_a_in <= `DISABLE;
	e_mar_b_in <= `DISABLE;
	e_addr_out <= `DISABLE;	
	e_calc_addr <= `DISABLE;
	e_mdr_a_in <= `DISABLE;
	e_mdr_b_in <= `DISABLE;	
	e_mdr_a_out <= `DISABLE;
	e_mdr_b_out <= `DISABLE;       
	e_calc_const <= `DISABLE;
	alu_op <= `ALU_NOP1;
	reg_a_sel <= 0;
	reg_b_sel <= 0;	
	e_data_in <= `DISABLE;
	e_data_out <= `DISABLE;
	bus_request <= `DISABLE;
	bus_in_use <= `DISABLE;
	bus_command <= `BUS_NONE;
	halt_computer <= `DISABLE;		
     end

   /* Konacni automat kontrolne jedinice. Na svakoj pozitivnoj ivici
      signala casovnika se, u zavisnosti od stanja automata, kao i 
      ulaznih signala (IR i PSW registri) aktiviraju odgovarajuci 
      signali i prelazi se u drugo stanje. */
   always @(posedge clk)
     begin
	// Iskljucimo najpre sve kontrolne signale (osim onih koji
	// kontrolisu  magistralu) 
	e_reg_a_in <= `DISABLE;
	e_reg_b_in <= `DISABLE;
	e_reg_a_out <= `DISABLE;
	e_reg_b_out <= `DISABLE;
	e_alu <= `DISABLE;
	e_psw_in <= `DISABLE;      
	e_ac_alu_in <= `DISABLE;
	e_ac_a_out <= `DISABLE;
	e_ac_b_out <= `DISABLE;
	e_pc_a_in <= `DISABLE;
	e_pc_b_in <= `DISABLE;
	e_pc_a_out <= `DISABLE;
	e_pc_b_out <= `DISABLE;
	e_pc_inc <= `DISABLE;	
	e_ir_a_in <= `DISABLE;
	e_ir_b_in <= `DISABLE;	
	e_mar_a_in <= `DISABLE;
	e_mar_b_in <= `DISABLE;
	e_calc_addr <= `DISABLE;
	e_mdr_a_in <= `DISABLE;
	e_mdr_b_in <= `DISABLE;	
	e_mdr_a_out <= `DISABLE;
	e_mdr_b_out <= `DISABLE;
	e_calc_const <= `DISABLE;
	reg_a_sel <= 0;
	reg_b_sel <= 0;	
	alu_op <= `ALU_NOP1;
	halt_computer <= `DISABLE;		
	
	case(state)
	  `FETCH:  // zapocinje dohvatanje instrukcije
	    begin
	       e_mar_a_in <= `ENABLE;
	       e_pc_a_out <= `ENABLE;
	       bus_request <= `ENABLE;	       
	       state <= `WAIT_BUS_INST;
	    end
	  `WAIT_BUS_INST:  // cekanje na magistralu
	    if(bus_grant)
	      begin
		 bus_request <= `DISABLE;
		 state <= `REQUEST_INST;		 		 
	      end
	  
	  `REQUEST_INST:  // slanje zahteva za citanje instrukcije
	    state <= `WAIT_INST;
	  
	  `WAIT_INST: // cekanje da instrukcija stigne
	    if(!bus_wait)
	      begin		 
		 state <= `DECODE;
	      end
	  
	  `DECODE: // dekodiranje instrukcije (prebacivanje u IR)
	    begin	       		    
	       e_ir_a_in <= `ENABLE;
	       e_mdr_a_out <= `ENABLE;		    		    
	       state <= `EXECUTE;	       
	    end 
	  `EXECUTE: // izvrsavanje instrukcije (zavisi od instrukcije u IR)
	    begin
	       case(ir_op_code)
		 `LDR0, `LDR1, `LDR2, `LDR3, `LDR4, `LDR5, `LDR6, `LDR7, `LDR8,
		 `LDR9, `LDR10, `LDR11, `LDR12, `LDR13, `LDR14, `LDR15:
		   begin
		      e_pc_a_out <= `ENABLE;
		      e_mdr_b_out <= `ENABLE;
		      e_calc_addr <= `ENABLE;
		      bus_request <= `ENABLE;		      
		      state <= `WAIT_BUS_LOAD;		      
		   end
	 	 `LDC0, `LDC1, `LDC2, `LDC3, `LDC4, `LDC5, `LDC6, `LDC7, `LDC8,
		   `LDC9, `LDC10, `LDC11, `LDC12, `LDC13, `LDC14, `LDC15:
		     begin
			e_calc_const <= `ENABLE;
			state <= `WRITE;		      
		     end
		 `LDI:
		   begin
		      reg_b_sel <= ir_src_reg;
		      e_reg_b_out <= `ENABLE;
		      e_mar_b_in <= `ENABLE;
		      bus_request <= `ENABLE;		      
		      state <= `WAIT_BUS_LOAD;		      
		   end		   
	 	 `STR0, `STR1, `STR2, `STR3, `STR4, `STR5, `STR6, `STR7, `STR8,
		   `STR9, `STR10, `STR11, `STR12, `STR13, `STR14, `STR15:
		     begin						
			e_pc_a_out <= `ENABLE;
			e_mdr_b_out <= `ENABLE;
			e_calc_addr <= `ENABLE;
			state <= `WRITE;			
		     end
		 `STI:
		   begin
		      reg_b_sel <= ir_src_reg;
		      e_reg_b_out <= `ENABLE;
		      e_mar_b_in <= `ENABLE;
		      state <= `WRITE;		      		      
		   end		 
		 `MOV, `ADD, `ADC, `SUB, `SHL, `SHR, `SAR, `AND, `OR, `XOR,
		   `INC, `DEC, `NEG, `NOT:
		   begin
		      reg_a_sel <= ir_dst_reg;
		      reg_b_sel <= ir_src_reg;
		      e_reg_a_out <= `ENABLE;
		      e_reg_b_out <= `ENABLE;
		      e_alu <= `ENABLE;	
		      e_ac_alu_in <= `ENABLE;
		      alu_op <= ir_op_code[3:0];		      
		      e_psw_in <= `ENABLE;
		      state <= `WRITE;		      		      		      
		   end
		 `CMP, `TST:
		   begin
		      reg_a_sel <= ir_dst_reg;
		      reg_b_sel <= ir_src_reg;
		      e_reg_a_out <= `ENABLE;
		      e_reg_b_out <= `ENABLE;
		      e_alu <= `ENABLE;	
		      alu_op <= ir_op_code == `CMP ? `ALU_SUB : `ALU_AND;		      
		      e_psw_in <= `ENABLE;
		      e_pc_inc <= `ENABLE;		      
		      state <= `FETCH;		      		      		      		      
		   end 
		 `JMP, `JE, `JNE, `JA, `JB, `JAE, `JBE, `JG, `JL, `JGE, `JLE:
		   begin
		      if(cond)
			begin
			   e_pc_b_in <= `ENABLE;
			   e_mdr_b_out <= `ENABLE;
			   state <= `FETCH;		      
			end
		      else
			begin
			   e_pc_inc <= `ENABLE;
			   state <= `FETCH;			   
			end
		   end 
		 `JMPI, `JEI, `JNEI, `JAI, `JBI, `JAEI, `JBEI, `JGI, `JLI, `JGEI, `JLEI:
		   begin
		      if(cond)
			begin
			   e_pc_a_in <= `ENABLE;
			   e_reg_a_out <= `ENABLE;
			   reg_a_sel <= ir_src_reg;			   
			   state <= `FETCH;		      
			end
		      else
			begin
			   e_pc_inc <= `ENABLE;
			   state <= `FETCH;			   
			end
		   end
		 `HALT:
		   begin
		      state <= `HALT_STATE;		      
		   end
	       endcase
	    end 
	  `WAIT_BUS_LOAD: // cekanje magistrale za citanje operanda
	    if(bus_grant)
	      begin
		 bus_request <= `DISABLE;
		 state <= `REQUEST_LOAD;		 		 
	      end
	  
	  `REQUEST_LOAD: // slanje zahteva za citanje operanda
	    state <= `WAIT_LOAD;
	  
	  `WAIT_LOAD: // cekanje operanda
	    if(!bus_wait)
	      begin		 
		 state <= `WRITE;
	      end

	  `WAIT_BUS_STORE:  // cekanje magistrale za upis rezultata u memoriju
	    if(bus_grant)
	      begin
		 bus_request <= `DISABLE;
		 state <= `REQUEST_STORE;		 		 
	      end 
	  `REQUEST_STORE:   // slanje zahteva za upis rezultata u memoriju
	    state <= `WAIT_STORE;	  
	  
	  `WAIT_STORE:  // cekanje na upis rezultata u memoriju
	    if(!bus_wait)
	      begin
		 e_pc_inc <= `ENABLE;		 
		 state <= `FETCH;
	      end	  
	  `WRITE:  // upis rezultata (u registar ili memoriju)
	    begin
	       case(ir_op_code)
		 `LDR0, `LDR1, `LDR2, `LDR3, `LDR4, `LDR5, `LDR6, `LDR7, `LDR8,
		 `LDR9, `LDR10, `LDR11, `LDR12, `LDR13, `LDR14, `LDR15:
		   begin
		      e_mdr_a_out <= `ENABLE;
		      reg_a_sel <= ir_op_code[3:0];
		      e_reg_a_in <= `ENABLE;
		      e_pc_inc <= `ENABLE;		      
		      state <= `FETCH;		      
		   end
		 `LDC0, `LDC1, `LDC2, `LDC3, `LDC4, `LDC5, `LDC6, `LDC7, `LDC8,
		   `LDC9, `LDC10, `LDC11, `LDC12, `LDC13, `LDC14, `LDC15:
		     begin
			e_mdr_a_out <= `ENABLE;		       
			reg_a_sel <= ir_op_code[3:0];
			e_reg_a_in <= `ENABLE;			
			e_pc_inc <= `ENABLE;
			state <= `FETCH;	       
		     end
		 `LDI:
		   begin
		      e_mdr_a_out <= `ENABLE;
		      reg_a_sel <= ir_dst_reg;
		      e_reg_a_in <= `ENABLE;
		      e_pc_inc <= `ENABLE;
		      state <= `FETCH;		      		      
		   end		
		 `STR0, `STR1, `STR2, `STR3, `STR4, `STR5, `STR6, `STR7, `STR8,
		   `STR9, `STR10, `STR11, `STR12, `STR13, `STR14, `STR15:
		     begin
			reg_a_sel <= ir_op_code[3:0];
			e_reg_a_out <= `ENABLE;
			e_mdr_a_in <= `ENABLE;
			bus_request <= `ENABLE;		      
			state <= `WAIT_BUS_STORE;		      
		     end
		 `STI:
		   begin
		      reg_a_sel <= ir_dst_reg;
		      e_reg_a_out <= `ENABLE;
		      e_mdr_a_in <= `ENABLE;
		      bus_request <= `ENABLE;		      
		      state <= `WAIT_BUS_STORE;		      
		     end		   
		 `MOV, `ADD, `ADC, `SUB, `SHL, `SHR, `SAR, `AND, `OR, `XOR,
		   `INC, `DEC, `NEG, `NOT:				   
		   begin
		      reg_a_sel <= ir_dst_reg;
		      e_reg_a_in <= `ENABLE;
		      e_ac_a_out <= `ENABLE;
		      e_pc_inc <= `ENABLE;
		      state <= `FETCH;	       
		   end
	       endcase
	    end 
	  `HALT_STATE:  // zavrsno stanje (iskljucuje se racunar)
	    halt_computer <= `ENABLE;	  
	endcase	
     end

   /* Signali za kontrolu magistrale se zbog efikasnije komunikacije na
   /* magistrali aktiviraju i deaktiviraju na negativnoj ivici
      casovnika. Naime, protokol magistrale je takav da se na silaznoj
      ivici salju kontrolni signali i adresa, a na uzlaznoj ivici slave
      uredjaj reaguje na zahtev. Na sledecoj silaznoj ivici master 
      prihvata podatke od slave-a (u slucaju operacije citanja). Na 
      ovaj nacin se efikasnije koristi ciklus, jer se koriste obe ivice
      casovnika. */
   always @(negedge clk)
     case(state)
       `REQUEST_INST, `REQUEST_LOAD:
	 begin
	    e_addr_out <= `ENABLE;
	    e_data_in <= `ENABLE;		 		 
	    bus_in_use <= `ENABLE;		 
	    bus_command <= `BUS_READ;
	 end
       `WAIT_INST, `WAIT_LOAD:	 
	 if(!bus_wait)
	   begin
	      e_addr_out <= `DISABLE;
	      e_data_in <= `DISABLE;		 		 
	      bus_in_use <= `DISABLE;		 
	      bus_command <= `BUS_NONE;
	   end       
       `REQUEST_STORE:
	 begin
	    e_addr_out <= `ENABLE;
	    e_data_out <= `ENABLE;		 		 
	    bus_in_use <= `ENABLE;		 
	    bus_command <= `BUS_WRITE;
	 end
       `WAIT_STORE:
	 if(!bus_wait)
	   begin
	      e_addr_out <= `DISABLE;
	      e_data_out <= `DISABLE;		 		 
	      bus_in_use <= `DISABLE;		 
	      bus_command <= `BUS_NONE;		 
	   end
     endcase
   
   /* DEBUG proces prikazuje informacije o promeni stanja. */
   always @(state)
     begin
	case(state)
	  `FETCH:
	    $display($time, ": state: FETCH");
	  `WAIT_BUS_INST:
	    $display($time, ": state: WAIT_BUS_INST");
	  `REQUEST_INST:
	    $display($time, ": state: REQUEST_INST");
	  `WAIT_INST:
	    $display($time, ": state: WAIT_INST");
	  `DECODE:
	    $display($time, ": state: DECODE");
	  `EXECUTE:
	    $display($time, ": state: EXECUTE");
	  `WRITE:
	    $display($time, ": state: WRITE");
	  `WAIT_BUS_LOAD:
	    $display($time, ": state: WAIT_BUS_LOAD");
	  `REQUEST_LOAD:
	    $display($time, ": state: REQUEST_LOAD");
	  `WAIT_LOAD:
	    $display($time, ": state: WAIT_LOAD");
	  `WAIT_BUS_STORE:
	    $display($time, ": state: WAIT_BUS_STORE");
	  `REQUEST_STORE:
	    $display($time, ": state: REQUEST_STORE");
	  `WAIT_STORE:
	    $display($time, ": state: WAIT_STORE");
	  `HALT_STATE:
	    $display($time, ": state: HALT_STATE");
	endcase
     end
   
endmodule // cu
